classdef PM1000

   properties
   
      model = "PM1000"
      version = "0.9.5"
      
   end

   methods(Static)
   
      %%%%%%%%%%%%%%%%%%%%%%%
      % Basic communication %
      %%%%%%%%%%%%%%%%%%%%%%%
      
      function isConnected = init(ARG)
         % PM1000.init'192.168.1.100');     opens TCP/IP device
         % PM1000.init'UDP192.168.1.100');  opens UDP device
         % PM1000.init(2);                  opens last USB device, with automatic re-connect, no user input
         % PM1000.init(1);                  opens last USB device
         % PM1000.init(0);                  shows list of USB devices and lets user chose one
         % PM1000.init;                     same as pm=PM1000(1);
         
         
         
         if (nargin<1)
            ARG=1; % connect to last device
         end;
      
         global PMHandle
         if length(ARG)>1&&sum(class(ARG)=='char')==4  % LAN DEVICE
            if sum(ARG(1:3)=='UDP')==3, % UDP DEVICE
               ARG = ARG(4:end);
               PMHandle = NOVUDP(ARG);
            else
               PMHandle = NOVTCP(ARG);
            end;
         else % USB DEVICE
            if ARG==2, % automatic re-connect in scripts
               if PMHandle.type=="USB"
                  pause(5);
                  tries=0;
                  while (PMHandle.isConnected==0)&&(tries<30),
                     pause(1);
                     tries = tries + 1
                     PMHandle = NOVUSB('LastDevPM.mat', ARG);
                  end;
               end;
            else
               PMHandle = NOVUSB('LastDevPM.mat', ARG);
            end;
         end;
         
         isConnected=PMHandle.isConnected;

      end;
      
      function close
         global PMHandle
         PMHandle.close; 
      end;
      
      function [res ok]  = read_raw(addr)
      
         global PMHandle
         
         [res ok] = PMHandle.read(addr);
         
         if ok==0, % su success in PM1000.reading. Try to connect again
            PM1000.init(2);
            [res ok] = PMHandle.read(addr);
         end;
         
      end;
      
      function [res ok]  = read(addr)
         for ii=1:length(addr)
            [res(ii) ~] = PM1000.read_raw(addr(ii));
         end;
      end;
      
      function [dout1 ok] = readburst(rdaddr, addrstart, addrstop, wraddr)
	  
         global PMHandle
         [dout1 ok] = PMHandle.readburst(rdaddr, addrstart, addrstop, wraddr);
         if ok==0, % su success in reading. Try to connect again
            PM1000.init(2);
            [dout1 ok] = PMHandle.readburst(rdaddr, addrstart, addrstop, wraddr);
         end;
		end;

      function ok = write_raw(addr, data)
         global PMHandle
         ok = PMHandle.write(addr, data);
         if ok==0, % su success in writing. Try to connect again
            PM1000.init(2);
            ok = PMHandle.write(addr, data);
         end;
      end;
      
      function ok = write(addr, data)
         if length(addr)==length(data),
            for ii=1:length(addr),
               ok = PM1000.write_raw(addr(ii), data(ii));
            end;
         else
            for ii=1:length(addr),
               ok = PM1000.write_raw(addr(ii), data(1));
            end;
         end;
      end;
      
      
      
      %%%%%%%%%%%%%%%%%%%%%%%%%%%
      % General instrument data %
      %%%%%%%%%%%%%%%%%%%%%%%%%%%
      
      function r = getfirmware % as string
         r = sprintf('%4X', PM1000.read(512+128));
      end

      function r = getserialnumber
         r = PM1000.read(512+133);
      end
      
      function r = getmoduletype % as string
         str=[];
         for ii=0:15,
            dummy = PM1000.read(512+144+ii);
            str=[ str char(bitshift(dummy, -8)) ];
            str=[ str mod(dummy, 2^8) ];
         end;
         r = str;
      end
      
      function r = getmacaddress % as string
         mac_address = PM1000.read(512+(248:253));
         r = sprintf("%02X:%02X:%02X:%02X:%02X:%02X", mac_address);
      end;
      
      function r = getipaddress % as string
         ip_address = PM1000.read(512+(239:242));
         r = sprintf("%d.%d.%d.%d", ip_address);
      end;
      
      function setipaddress(ipstring) % as string
         ip_address = split(ipstring, '.');
         for ii=1:4,
            PM1000.write(512+239+ii-1, str2num(ip_address(ii)));
         end;
         % to store the ip addres permanently, write BIT1 of register 512+65
         dummy = PM1000.read(512+65);
         PM1000.write(512+65, bitor(dummy, 2^1));
      end;
      
      
      %%%%%%%%%%%%%%%%%%%%%%%%%
      % Calibration functions %
      %%%%%%%%%%%%%%%%%%%%%%%%%
      
      function setATE(ATE)
         PM1000.write(512+1, ATE);
      end;
      
      function r = getATE
         r = PM1000.read(512+1);
      end;
      
      function setnormmode(norm_mode)
         % 0: Non-normalized
         % 1: Standard normalization
         % 2: Exact normalization3
         norm_mode=min(2, max(0, norm_mode));
         PM1000.write(512+46, norm_mode);
      end;
      
      function r = getnormmode
         r = PM1000.read(512+46);
      end;
      
      function setnormlevel(level_uW) % in microwatts
         %for example: PM1000.write(512+38, 2^11); % Normalize S1-S3 to 2^11=2048 µw
         PM1000.write(512+38, round(level_uW));
      end;
      
      function r = getnormlevel % in microwatts
         r = PM1000.read(512+38);
      end;
      
      
      
      function SetNonNormPowerRef(level_uW) % in microwatts       
         LevelExp = ceil(log2(level_uW));
         PM1000.write(512+38, 2^LevelExp);
         PM1000.write(512+74, 16-LevelExp);
      end;
      
      function r = getoptfrequency % in THz
         r = PM1000.read(512+69)/100;
      end;
      
      function setoptfrequency(frequency) % in THz
         r = PM1000.write(512+69, round(frequency*100));
      end;
      
      function r = getmincalfreq % in THz
         r = PM1000.read(512+70)/100;
      end;

      function r = getmaxcalfreq % in THz
         r = PM1000.read(512+71)/100;
      end;
      
      function r = getminfreq % in THz
         r = PM1000.read(512+67)/100;
      end;

      function r = getmaxfreq % in THz
         r = PM1000.read(512+68)/100;
      end;
      
      function setswitchstate(state)
         PM1000.write(512+430, state);
      end;
      
      function setpowerleftshift(bits)
         PM1000.write(512+74, bits);
      end;
      
      function r = getpowerleftshift
         r = PM1000.read(512+74);
      end;
      
      function r = getswitchstate
         r = PM1000.read(512+430);
      end;
      
      function r = gettemperature
         r = PM1000.read(512+421)/16;
      end;
      
      function settemperature(tmp)
      
         % setpoint of the heating element
         PM1000.write(512+431, round(tmp*16));
         % enable the heating element
         PM1000.write(512+432, 59362);
         
         % setpoint (max. 63°C) and hysteresis for the cooling fan
         hyst = 0;
         %temp = 63;
         temp = tmp+1;
         PM1000.write(512+433, round(hyst*16)*2^10 + round(temp*16));
      end;
      
      
      function setsensmode(mode)
         switch mode
            case 0 % normal mode
               PM1000.write(512+196, bin2dec('00001'));
            case 1 % sensitive mode (optional)
               PM1000.write(512+196, bin2dec('00000'));
            case 2 % ultrasensitive mode (optional)
               PM1000.write(512+196, bin2dec('10000'));
            otherwise % automatic
               PM1000.write(512+196, bin2dec('00010'));
         end;
      end;
      
      function mode = getsensmode % if set to automatic
         val = dec2bin(PM1000.read(512+196), 16);
         code = bin2dec(val(end-3:end-2));
         switch code
            case 1 % normal mode
               mode = 0;
            case 0 % sensitive mode
               mode = 1;
            case 2 % ultrasensitive mode
               mode = 2;
            otherwise % undefined mode
               mode = -1;           
         end;
      end;
      
      function setdarkcurrent
      
         CurrentPow_FPGA = PM1000.getPower;
         CurrentATE_FPGA = PM1000.getATE;
         
         if CurrentPow_FPGA>10,
            disp("Please turn off optical power. Press Enter to proceed...");
            pause;
         end;
         
         PM1000.setATE(23);
         
         for sensmode=0:2
         
            PM1000.setsensmode(sensmode);
            
            % set dark current registers to zero for measurement
            for ii=0:3
               PM1000.write(512 + 164 + sensmode*100 + ii, 0);
            end;
            
            pause(.2);
            
            % read photocurrent values and store in dark current registers
            for ii=0:3
               DarkCur = PM1000.read(512 + 160 + ii)
               PM1000.write(512 + 164 + sensmode*100 + ii, DarkCur);
            end;
            
         end;
         
         % restore the ATE value
         PM1000.setATE(CurrentATE_FPGA);
         % set to normal sens mode
         PM1000.setsensmode(0);
         
      
      end;
      
      
      
      %%%%%%%%%%%%%%%%%%%%%%%%%%%%%
      % Data aquisition functions %
      %%%%%%%%%%%%%%%%%%%%%%%%%%%%%
      
      
      function r = getphotocurrents
         r=zeros(4,1);
         r(1)=PM1000.read(512+2);
         r(2)=PM1000.read(512+6); % latched signals
         r(3)=PM1000.read(512+7); % latched signals
         r(4)=PM1000.read(512+8); % latched signals
         bitshift=PM1000.read(512+9);
         r=r/2^bitshift;
      end;
      

      
      function s = getStokes(norm)
         s=zeros(4,1);
         switch norm
            case 0 % Non-normalized
               s(1)=PM1000.read(512+38)/2^15; % power reference for S1,S2,S3
               s(2)=(PM1000.read(512+42)-2^15)/2^15;
               s(3)=(PM1000.read(512+43)-2^15)/2^15;
               s(4)=(PM1000.read(512+44)-2^15)/2^15;
            case 1 % Standard
               s(1)=PM1000.read(512+24)/2^15; % DOP
               s(2)=(PM1000.read(512+28)-2^15)/2^15;
               s(3)=(PM1000.read(512+29)-2^15)/2^15;
               s(4)=(PM1000.read(512+30)-2^15)/2^15;
            case 2 % Exact
               s(1)=PM1000.read(512+31)/2^15; % DOP
               s(2)=(PM1000.read(512+35)-2^15)/2^15;
               s(3)=(PM1000.read(512+36)-2^15)/2^15;
               s(4)=(PM1000.read(512+37)-2^15)/2^15;
            case 3 % In µW
               s(1)=PM1000.read(512+10) + PM1000.read(512+11)/2^16;
               g = PM1000.read(512+18)-2^15;
               n = bitxor(PM1000.read(512+19), 2^15)/2^16;
               if (g<0), s(2)=g-n; else s(2)=g+n; end;
               g = PM1000.read(512+20)-2^15;
               n = bitxor(PM1000.read(512+21), 2^15)/2^16;
               if (g<0), s(3)=g-n; else s(3)=g+n; end;
               g = PM1000.read(512+22)-2^15;
               n = bitxor(PM1000.read(512+23), 2^15)/2^16;
               if (g<0), s(4)=g-n; else s(4)=g+n; end;
         end
      end;
      
      function r = getDOP
         r = PM1000.read(512+24)/2^15;
      end;
      
      function r = getPower
         r = PM1000.read(512+10) + PM1000.read(512+11)/2^16; % Power in µW
      end;
      
      function ev = getEigen(sel)
         % sel:
         % 1 = eigenvector1, corresp. to strongest eigenvalue
         % 2 = eigenvector2, corresp. to weakest eigenvalue
         % 3 = eigenvector3, cross product of eigenvector1 and eigenvector2
         % 4 = mean Stokes vector
         if (sel>=1)&&(sel<=4)
            ev = zeros(3,1);
            for ii=1:3,
               val = PM1000.read(512+320+ii-1+(sel-1)*3);
               % bit vector to signed
               if val>=2^15, val=val-2^16; end;
               ev(ii) = val;
            end;
            
            % normalization
            ev = ev / norm(ev);
         else
            ev = 0;
         end;
      end;
      
      function r = getMatrix(regoffset)
      
         for xdim=1:4,
            for ydim=1:4,
               matrix(xdim, ydim) = (PM1000.read(512+regoffset-1+xdim+(ydim-1)*4)) -2^15;
            end;
         end;
         
         r = matrix;
      
      end;
      
      function setUserMatrix(MMinv, MMinv_shift)
         regoffset = 512+48;
         for y = 1:4
            for x = 1:4
               data = round(MMinv(x,y)) + 2^15;
               addr = regoffset + (y-1)*4 + (x-1);
               PM1000.write(addr, data);
            end;
         end;
         PM1000.write(regoffset+16, MMinv_shift);
         
         % use external calibration matrix
         PM1000.write(512+65, 1);
      end;
      
      %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
      % SDRAM recording & data transfer %
      %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
      
      function pclear % clears poincare sphere on hdmi output
         PM1000.write(512+189, 0);
      end;
      
      function setME(ME) % sets Memory Exponent. 
         PM1000.write(512+73, ME);
      end;
      
      function [me, maxme] = getME
         res = PM1000.read(512+73);
         me = mod(res, 2^8);
         maxme = bitshift(res, -8);
      end;
      
      function r = isBusy
         r = bitand(PM1000.read(512+72), 1);
      end;
      
      function waitBusy
         busy = 1;
         while busy>0
            busy = PM1000.isBusy;
         end;
      end;
      
      function r = getrecaddress
        addr = PM1000.read(512+76);
        addr = addr + (PM1000.read(512+77) * 2^16);
        r = addr;
      end;
      
      function r = getrecblock
        r = PM1000.read(512+78);
      end;
      
      function res = recsdram(ATE, ME);
         
         % cancel any ongoing measurement
         if PM1000.isBusy,
            PM1000.write(512+72, 1);
         end;
         
         PM1000.setATE(ATE);
         PM1000.setME(ME);
         
         syncextrising = 0;
         syncextfalling = 0;
         cyclic = 0;
         trigonnextbnc = 0;
         PM1000.write(512+72, trigonnextbnc*2^7 + syncextfalling*2^6 + syncextrising*2^5 + cyclic*2^2 + ~trigonnextbnc); % trigger recording
         
         PM1000.waitBusy;
         
         res = PM1000.getsdram_norm(0, 2^ME, 1);
      end;
         
         

      
      function [res, ok, status] = getsdram(startaddr, numaddr)
      
         global PMHandle
         [res, ok, status] = PMHandle.readhspm(startaddr, numaddr);
         
         while (ok==0),
            PM1000.init(2);
            [res ok]  = PM1000.read(512);
            if (ok==1),
               [res, ok, status] = PMHandle.readhspm(startaddr, numaddr); 
            end;
         end;
      end;
      
      function [res, ok, status] = getsdram_norm(startaddr, numaddr, normalization)
         [res, ok, status] =  PM1000.getsdram(startaddr, numaddr);
         if normalization==0,
            for ii=1:size(res, 2),
               res(2,ii) = res(2,ii)*2 - 2^16;
               res(3,ii) = res(3,ii)*2 - 2^16;
               res(4,ii) = res(4,ii)*2 - 2^16;
            end;
         elseif normalization==1,
            for ii=1:size(res, 2),
               res(1,ii) = res(1,ii) / 2^15;
               res(2,ii) = (res(2,ii)-2^15) / 2^15;
               res(3,ii) = (res(3,ii)-2^15) / 2^15;
               res(4,ii) = (res(4,ii)-2^15) / 2^15;
            end;
         end;
      end;
      
      
      % Reads one block of the Poincaré sphere frame buffer
      function [res ok] = readfr(BlockNr)
         global PMHandle
         [res, ok] = PMHandle.readfr(BlockNr);
      end;
	  
	  function setSOPstreamperiod(sr) % in microseconds
         PM1000.write(512+109, sr);
      end;
      
      function r = getSOPstreamperiod % in microseconds
         r = PM1000.read(512+109);
      end;
      
	  function setSOPstream(enable)
         % only available with USB 2.0 and 3.0
         global PMHandle
         PMHandle.setsopstream(enable);
      end;
	  
	  function r = getSOPstream(norm) % norm = normalization
         % only available with USB 2.0 and 3.0
         global PMHandle
         r = PMHandle.getsopstream(norm);
      end;
      
      
      function histogramstop
         %global PMHandle
         %PMHandle.histogramstop;
         PM1000.write(512 + 232, 2);
      end;
      
      function histogramstart(bshift, clkdiv, tau) 
         global PMHandle
         %PMHandle.histogramstart(3,1,1);
         PMHandle.histogramstart(bshift,clkdiv,tau);
      end;
      
      function r=gethistogram(HistVexp, HistClkDiv, HistTau, time_s)
      
         % histogramstop
         PM1000.write(512 + 232, 2);
         
         %HistVexp = 3;
         %HistClkDiv = 7;
         %HistTau = 6;
         
         % histogramstart
         PM1000.write(512 + 234, HistVexp);
         PM1000.write(512 + 236, HistClkDiv * 2 ^ 6 + HistTau - 1);
         PM1000.write(512 + 232, 1);
         
         pause(time_s);
      
         % histogramstop
         PM1000.write(512 + 232, 2);
      
         % gethistogram
         histhelp1 = PM1000.readburst(512 + 230, 2 ^ 15 + 0, 2 ^ 15 + 1023, 512 + 231);
         histhelp2 = PM1000.readburst(512 + 230, 2 ^ 14 + 0, 2 ^ 14 + 1023, 512 + 231);
         histhelp3 = PM1000.readburst(512 + 230, 0,          1023,          512 + 231);
                
         for ii=1:1024
            r(ii) = histhelp1(ii) * 2 ^ 32 + histhelp2(ii) * 2 ^ 16 + histhelp3(ii);
         end;

      end;
      
      function setppssync(enable, config)
         % requires firmware >= 1.1.0.0 and PM1000 main board with AMD Artix 7 FPGA
         % arg 1: 1 (enable) or 0 (disable)
         % arg 2: tbd
         if (nargin<2) % don't modify configuration, 
            conf = PM1000.read(512+211);
            if enable==0,
               PM1000.write(512+211, bitand(conf, hex2dec('FFFE')));
            else
               PM1000.write(512+211, bitor(conf, 1));
            end;
         else
            PM1000.write(512+211, conf * 2^1 + enable)
         end; 
       end;
       
       function [timeout shift_period deviation_ns] = getppssyncstatus
       
         conf = mod(PM1000.read(512+211), 2^9); 
         
         PM1000.write(512+211, 0*2^9 + conf);
         timeout = PM1000.read(512+212)
         
         PM1000.write(512+211, 1*2^9 + conf);
         shift_period = PM1000.read(512+212) % in ns*10
         
         PM1000.write(512+211, 2*2^9 + conf);
         deviation = PM1000.read(512+212);
         if deviation>2^15, deviation=deviation-2^16; end;
         deviation_ns = 20*deviation
       end;
         
      
      
      
      
	  
	  
	  



   end;

end