classdef NOVTCP < handle

   properties
   
      type = "UDP";
      version = "1.1.1"
      IPADDR
      Handle
      isConnected = 0

   end


   methods
   
      function obj = NOVTCP(IPADDR)
      
         %disp(sprintf('IPADDR = %s;', IPADDR));
         obj.IPADDR = IPADDR;
         init(obj);
         
      end;
      
      function s = getip(obj)
         s = obj.IPADDR;
      end;

      function init(obj)
      
         if exist('OCTAVE_VERSION', 'builtin') % Octave
            pkg load instrument-control
         end;
            
         try
            obj.Handle = tcpclient(obj.IPADDR, 5025);
         catch ME
            obj.isConnected = 0;
            warning("Error using tcpclient:")
            disp(ME.message)
            return
         end;
         obj.Handle.ByteOrder = "big-endian"; % works only in Matlab so far
         
         obj.isConnected = 1;

      end;
      
      function close(obj)
         obj.Handle = 0;
         clear obj.Handle;
         obj.isConnected = 0;
      end;
      
      function ok = write(obj, addr, data)
      
         seq1 = [  87 bitshift(addr, -8) mod(addr, 2^8) bitshift(data, -8) mod(data, 2^8)];

         write(obj.Handle, char(seq1));
         
         ok = 1;


      end;
      
      function [res ok] = read(obj, addr)
      
         debugmode=0;

         ok=0;

         if length(addr)==1,
            seq1 = [ 82 bitshift(addr, -8) mod(addr, 2^8)];
         else
            seq1 = [ 77 length(addr) ];
            for ii=1:length(addr),
               seq1 = [ seq1 bitshift(addr(ii), -8) mod(addr(ii), 2^8)];
            end;
         end;

         BytesAvailable = obj.Handle.NumBytesAvailable;
         
         if BytesAvailable,
            A = read(obj.Handle, BytesAvailable);
         end;

         tic
         
         write(obj.Handle, char(seq1));

         BytesAvailable=0;
         StopCounter=0;
         while BytesAvailable<2*length(addr) && StopCounter<2000,
            BytesAvailable = obj.Handle.NumBytesAvailable;
            StopCounter = StopCounter+1;
            pause(.001);
         end;

         if debugmode
            toc % 10..200 ms with PM1000
         end;

         res=zeros(length(addr), 1);
         if BytesAvailable>=2*length(addr),
            A = read(obj.Handle, BytesAvailable);
            for ii=1:length(addr),
               res(ii) = uint16(A(2*(ii-1)+1))*2^8 + uint16(A(2*(ii-1)+2));
               ok=1;
            end;
         else
            BytesAvailable
            StopCounter
         end;

         if debugmode
            BytesAvailable
         end
         
      end;
      
      
      
      % only for PM1000: Reads one block of the Poincaré sphere frame buffer
      function [res ok] = readfr(obj, BlockNr)
      
         TCPHandle = obj.Handle;
      
         NumBytes = 2^12; % one complete frame

         debugmode=0;

         ok=0;

         tic

         % request data stream
         write(obj, 512 + 105, BlockNr);
         pause(.001);
         write(obj, 512 + 104, 27273);
         pause(.001);


         BytesAvailable=0;
         StopCounter=0;
         while BytesAvailable<NumBytes & StopCounter<2000,
            BytesAvailable = TCPHandle.NumBytesAvailable;
            StopCounter = StopCounter+1;
            pause(.001);
         end;

         if debugmode
            toc % 10..200 ms with PM1000
         end;


         res=0;
         if BytesAvailable>=NumBytes,
            res = fread(TCPHandle, BytesAvailable);
            ok=1;
         else
            BytesAvailable
            StopCounter
         end;

         if debugmode
            BytesAvailable
         end
      end;
      
      
      % only for PM1000: Reads data from SDRAM
      function [res, ok, status] = readhspm(obj, startaddr, numaddr)
      
         status = 0;
         
         buffersizebytes=2^14;


         %cycles=max(1, min(32, numaddr*8/buffersizebytes))
         %cycles=max(1, min(32, numaddr*8/buffersizebytes)) % debug
         cycles=max(1, min(32, ceil(numaddr*8/buffersizebytes))); % debug
         cycles=2^ceil(log2(cycles));
         
         numaddr=ceil(numaddr/cycles)*cycles;


         buffersizeaddr=buffersizebytes/8*cycles;

         packets_transferred=0;

         res=[];

         tic;
            
         while packets_transferred<numaddr,
            
             packetsinthissequence=min(buffersizeaddr, numaddr-packets_transferred);
            
             startaddrseq = startaddr + packets_transferred;

             [res_seq, ok, timeoutreached] = PM1000_requestpackets(obj, startaddrseq, packetsinthissequence, cycles);
             
             if timeoutreached,
               break;
             end;
            
             if ok,
                 packets_transferred=packets_transferred+packetsinthissequence;
                 res=[res res_seq];
             else
                 disp('HS data transmission failed. Trying again...')
             end;
             
             
                 

         end;

         %toc
            
         %pause(.1); BytesAvailable = TCPHandle.NumBytesAvailable   

      end
      
      
      
      % only for PM1000: Requests one or more packets of data from SDRAM
      function [res, ok, timeoutreached] = PM1000_requestpackets(obj, startaddrseq, packetsinthissequence, cycles)

         TCPHandle = obj.Handle;
         

         % reqisters for HS-configuration
         addr_data = [ 512+105 mod(startaddrseq, 2^16);
                       512+106 bitshift(startaddrseq, -16) + log2(cycles)*2^12;
                       %512+107 packetsinthissequence;
                       512+107 round(packetsinthissequence/cycles);
                       512+104 39294]; % trigger code for HS-transfer over LAN
         for ii=1:4,
              seq(:,ii) = [87 bitshift(addr_data(ii,1), -8) mod(addr_data(ii,1), 2^8) bitshift(addr_data(ii,2), -8) mod(addr_data(ii,2), 2^8) ];
         end;
         
         %writeepstcp(512+105,0); % set address: lower 16 bits
         
         % send the request
         write(TCPHandle, char(reshape(seq, 1, 20))); % writing as one packet improves timing compared to single write packets
         
         timeouttic=tic;
         
         res=[];
         for ii=1:cycles,
            BytesAvailable=0;
            timeoutreached=0;
            while BytesAvailable<packetsinthissequence*8/cycles && timeoutreached==0,
                 pause(.001);
                 BytesAvailable = TCPHandle.NumBytesAvailable;
                 if toc(timeouttic)>2
                     timeoutreached=1;
                     disp('Timeout reached before data received!');
                     break;
                 end;
            end;
            
            if timeoutreached,
               break;
            end;
            %toc(timeouttic) % 17 ms for 512 addr
            %toc(timeouttic) % 622 ms for 2^16 addr
            
            if BytesAvailable>=packetsinthissequence*8/cycles, % all bytes in buffer
                 
               if exist('OCTAVE_VERSION', 'builtin') % Octave
                  B = read(TCPHandle, packetsinthissequence*8/cycles); % 2 Bytes per Sample
                  for ii=1:packetsinthissequence*4/cycles, % reverse byte order as long as Octave does not support big-endian
                     A(ii) = uint16(B((ii-1)*2+1))*256 + uint16(B((ii-1)*2+2));
                  end;
               else
                  A = read(TCPHandle, packetsinthissequence*4/cycles, 'uint16'); % 2 Bytes per Sample
               end;
                 res=[res reshape(A, 4,packetsinthissequence/cycles)];
                 ok=1;
             else
                 ok=0;
            end;
         end;
      end





         
         

   end;

end