classdef LU1000

   properties
   
      model = "LU1000"
      version = "0.9.0"
      numoflasers
      
      % Laser1:
      firstfreq
      lastfreq
      grid
      maxchannels
      startfreq

   end


   methods(Static)
   
      %%%%%%%%%%%%%%%%%%%%%%%
      % Basic communication %
      %%%%%%%%%%%%%%%%%%%%%%%
      
      function init(ARG)
         % LU1000.init('192.168.1.100');    opens LAN device
         % LU1000.init(1);                  opens last USB device
         % LU1000.init(0);                  shows list of USB devices and lets user chose one
         % LU1000.init;                     same as LU1000.init(1);
         
         if (nargin<1)
            ARG=1; % connect to last device
         end;
      
         global LUHandle
         if length(ARG)>1&&sum(class(ARG)=='char')==4  % LAN DEVICE
            LUHandle=NOVTCP(ARG);
         else % USB DEVICE
            LUHandle=NOVUSB('LastDevLU.mat', ARG);
         end;
         
         %LU1000.numoflasers = LU1000.getnumoflasers;
         %LU1000.firstfreq = LU1000.getfirstfreq(1);
         %LU1000.lastfreq = LU1000.getlastfreq(1);
         %LU1000.startfreq = LU1000.getstartfreq(1);
         %LU1000.grid = LU1000.getgrid(1);
         %LU1000.maxchannels = floor((LU1000.lastfreq-LU1000.firstfreq)/LU1000.grid*1000+1);
         
      end;
      
      function close
         global LUHandle
         LUHandle.close; 
      end;
      
      function [res ok] = read(addr)
      
         global LUHandle;
         
         [res ok] = LUHandle.read(addr);
         
         if ok==0, % no success in reading. Try to connect again
            LU1000.init;
            [res ok] = LUHandle.read(addr);
         end;
         
      end;

      function ok = write(addr, data)
         global LUHandle
         ok = LUHandle.write(addr, data);
         if ok==0, % no success in writing. Try to connect again
            LU1000.init;
            [res ok] = LUHandle.write(addr, data);
         end;
         
         % pause 20 ms to be sure internal RS232-conversion to ITLA-interface is completed
         mstic=tic;
         mspause=0;
         while mspause<0.02,
            mspause=toc(mstic);
         end;
         
      end;
      
      %%%%%%%%%%%%%%%%%%%%%%%%%%%%%
      % Device specific functions %
      %%%%%%%%%%%%%%%%%%%%%%%%%%%%%
      
      function r = getnumoflasers
         r = LU1000.read(2);
      end
      
      function r = getboardtemperature % sensor 1, in °C
         r = LU1000.read(51)/16;
      end

      function r = getboardtemperature2 % sensor 2 (optional), in °C
         r = LU1000.read(52)/16;
      end
      
      function r = getfirmware % as string
         r = sprintf('%4X', LU1000.read(64));
      end

      function r = getserialnumber
         r = LU1000.read(65);
      end
      
      function r = getmoduletype % as string
         str=[];
         for ii=0:15,
            dummy = LU1000.read(68+ii);
            str=[ str char(bitshift(dummy, -8)) ];
            str=[ str mod(dummy, 2^8) ];
         end;
         r = str;
      end
            
      function r = getmacaddress % as string
         mac_address = LU1000.read(121:126);
         r = sprintf("%02X:%02X:%02X:%02X:%02X:%02X", mac_address);
      end;

      %%%%%%%%%%%%%%%%%%%%%%%%%%%%
      % Laser specific functions %
      %%%%%%%%%%%%%%%%%%%%%%%%%%%%

      
      function setenable(laserno, onoff) % >0=on, 0=off
         if onoff,
            LU1000.write(laserno*128+50, 8);
         else
            LU1000.write(laserno*128+50, 0);
         end
      end
      
      function r = getenable(laserno)
         r = LU1000.read(laserno*128+50);
      end
      
      function setpower(laserno, power) % in dBm
         LU1000.write(laserno*128+49, round(power*100)); % in dBm*100
      end
      
      function r=getpower(laserno) % in dBm
         r = LU1000.read(laserno*128+49)/100; % in dBm*100
      end
      
      function setfreq(laserno, freq) % in THz
         channel = round((freq-obj.firstfreq)/obj.grid*1000+1);
         if channel>=1 && channel<=obj.maxchannels,
            LU1000.write(laserno*128+48, channel);     
         else
            disp('   ---- Error: Freq out of range!  ----');
         end
      end
      
      function r= getfreq(laserno) % in THz
         r = LU1000.read(laserno*128+64) + LU1000.read(laserno*128+65)/10000;
      end
      
      function r= getlasertemperature(laserno) % in °C
         r = LU1000.read(laserno*128+67)/100;
      end
      
      function r=getgrid(laserno) % in GHz
         r = LU1000.read(laserno*128+52)/10;
      end;
      
      function setgrid(laserno, grid) % in GHz
         LU1000.write(laserno*128+52, round(grid*10));
      end
      
      function setchannel(laserno, channel)
         LU1000.write(laserno*128+48, channel);     
      end
      function r=getchannel(laserno)
         r = LU1000.read(laserno*128+48);
      end;
      
      function r=getstartfreq(laserno) % in THz
         r = LU1000.read(laserno*128+53) + LU1000.read(laserno*128+54)/10000;
      end;
      
      function r=setstartfreq(laserno, freq) % in THz
         LU1000.write(laserno*128+53, floor(freq));
         LU1000.write(laserno*128+54, round(mod(freq, 1)*10000));
      end;
      
      function r=getfirstfreq(laserno) % in THz
         r = LU1000.read(laserno*128+82) + LU1000.read(laserno*128+83)/10000;
      end;
      
      function r=getlastfreq(laserno) % in THz
         r = LU1000.read(laserno*128+84) + LU1000.read(laserno*128+85)/10000;
      end;
      
      function r=getmingrid(laserno) % in GHz
         r = LU1000.read(laserno*128+86)/10;
      end;

      function r=getminpower(laserno) % in dBm
          r = LU1000.read(laserno*128+80)/100; % in dBm*100
      end;

      function r=getmaxpower(laserno) % in dBm
          r = LU1000.read(laserno*128+81)/100; % in dBm*100
      end;
      
      function setfinetune(laserno, tune) % in MHz
         tune = round(tune);
         if tune<0, tune = 2^16+tune; end;
         LU1000.write(laserno*128+98, tune);     
      end
      
      function r=getfinetune(laserno) % in MHz
          r = LU1000.read(laserno*128+98);
      end;
      
      function setwhispermode(laserno, whisper) % 0=off, 2=on
         LU1000.write(laserno*128+108, whisper);     
      end
      
      function r=getwhispermode(laserno) % 0=off, 2=on
          r = LU1000.read(laserno*128+108);
      end;
      

   end;

end